/************************************************************************************
* Implements Timeout/Trigger event timers
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Includes
*************************************************************************************
************************************************************************************/

#include "EmbeddedTypes.h"
#include "PortConfig.h"
#include "Interrupt.h"
#include "Phy.h"



/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

static volatile phyEvent32BitTimer_t mPhyEventTimeoutTimer;
static volatile phyEvent32BitTimer_t mPhyEventTriggerTimer;
static volatile phyEvent32BitTimer_t mPhyCurrentTime;

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/******************************************************************************
*Function name: PhyTimeReadClock
*
*Parameters passed in: uint32_t * pTime
*
*Return type: none
*
*Prototype: void PhyTimeReadClock(uint32_t * pTime);
*
*Description: Used for reading the current time. The current time represents 
*             the current value of a 32bit free running counter. This 32 bit free running 
*             counter is implemented  using a hardware MCU 16 bit free running counter 
*             plus a software 16 bit counter which is incremented when MCU free running 
*             counter reach the modulo value 0xFFFF. 
*             The free running counter resolution is 1 u-second.
******************************************************************************/

void PhyTimeReadClock(phyEvent32BitTimer_t * pTime)
{
  uint32_t CCR;

#if defined (IAR)
  CCR= IntDisableAll();	
#elif  defined (CW)
  IntDisableAll(CCR);		
#endif
  

  pTime->byteAccess[2] = (uint8_t) (gPhyTimerCnt<<8);
  pTime->byteAccess[3] = (uint8_t) gPhyTimerCnt;
  pTime->wordAccess[0] = mPhyCurrentTime.wordAccess[0];
  if(gPhyTimerCtrl & gTimerCtrl_TOF_c)
  {
    gPhyTimerCtrl |=  gTimerCtrl_TOF_c;                                         //Clean TOF Flag
    mPhyCurrentTime.wordAccess[0]++;

  pTime->byteAccess[2] = (uint8_t) (gPhyTimerCnt<<8);
  pTime->byteAccess[3] = (uint8_t)  gPhyTimerCnt;
    pTime->wordAccess[0] = mPhyCurrentTime.wordAccess[0];
  }
  IntRestoreAll(CCR);
}

/******************************************************************************
*Function name: PhyTimeSetEventTimeout
*
*Parameters passed in: uint32_t endTime
*
*Return type: none
*
*Prototype: void PhyTimeSetEventTimeout(uint32_t endTime);
*
*Description: This function is used to set a relative 32 bit timeout (startTime)
*             for a PHY request action. This function performs the following actions: 
*             read the current time using PhyTimeReadClock(), add the relative timeout 
*             to the current time resulting an 32 bit absolute time, program the 
*             hardware MCU 16 bit comparator with the lower part of the absolute time, 
*             program the 16 bit software comparator with la higher part of the 
*             absolute time and enable the interrupt request in the corresponding 
*             TPM channel status and control register. 
*             When the 32 bit free running counter reach the 32 bit absolute time 
*             the current PHY requested action is aborted and a timeout indication function 
*             is called to inform the upper layer.
******************************************************************************/

void PhyTimeSetEventTimeout(uint32_t EndTime)
{
  
  uint16_t absTime;             
  uint16_t currentTime;        
  
  phyEvent32BitTimer_t * pEndTime = (phyEvent32BitTimer_t * )&EndTime;
  
  gPhyEventTimeoutCtrl &= ~(gEventTimerCtrl_IE_c);                              //Disable CHIE (C1SC) //@CMA ORIGINAL LINE
    
  currentTime = gPhyTimerCnt;

  absTime = (currentTime + pEndTime->wordAccess[1]);
  
  gPhyEventTimeoutCmp = absTime;

#if defined(IAR)   
  mPhyEventTimeoutTimer.wordAccess[1] = (uint16_t) absTime;                     
#endif
  
#if defined (CW)  
  mPhyEventTimeoutTimer.wordAccess[1] = *((uint16_t *) absTime);                 
#endif
  mPhyEventTimeoutTimer.wordAccess[0] = pEndTime->wordAccess[0];
  
  currentTime = gPhyEventTimeoutCmp;

  //TODO !!!!!!!!!!!  
  while(currentTime !=  absTime)
  {
    currentTime = gPhyEventTimeoutCmp;
    
  }
  
  if(gPhyEventTimeoutCtrl & gEventTimerCtrl_F_c)                                //If TPM_CH1F is set
  {
    gPhyEventTimeoutCtrl |= gEventTimerCtrl_F_c;                                //Clear Flag
    
  }
  gPhyEventTimeoutCtrl |= (gEventTimerCtrl_IE_c);                               //Enable TPM_CH1IE
  //PTAD |= (uint8_t )(0x01);
}

/******************************************************************************
*Function name: PhyTimeSetEventTrigger
*
*Parameters passed in: uint32_t startTime
*
*Return type: none
*
*Prototype: void PhyTimeReadClock(uint32_t startTime);
*
*Description: This function is used to start a PHY request action after a relative 32 bit delay (startTime). 
*             This function performs the following actions: read the current time using PhyTimeReadClock(), 
*             add the relative delay to the current time resulting an 32 bit absolute time, 
*             program the hardware MCU 16 bit comparator with the lower part of the absolute time, 
*             program the 16 bit software comparator with la higher part of the absolute time 
*             and enable the interrupt request in the corresponding TPM channel status and control register. 
*             When the 32 bit free running counter reach the 32 bit absolute time the current 
*             PHY requested action is started by putting the transceiver in transmitter mode 
*             or in receiver mode depending on the PHY state gTX_c respectively gRX_c/gCCA_c/gED_c.
******************************************************************************/

void PhyTimeSetEventTrigger(uint32_t StartTime)
{
  
  uint16_t absTime;             
  uint16_t currentTime;         
  
  phyEvent32BitTimer_t * pStartTime = (phyEvent32BitTimer_t * )&StartTime;
  
  gPhyEventTriggerCtrl |= gEventTimerCtrl_IE_c;
    
  currentTime = gPhyTimerCnt;

  
   absTime = (currentTime + pStartTime->wordAccess[1]);
  
  gPhyEventTriggerCmp = absTime;

    
  mPhyEventTriggerTimer.wordAccess[1] = absTime;
  mPhyEventTriggerTimer.wordAccess[0] = pStartTime->wordAccess[0];
  
  currentTime = gPhyEventTriggerCmp;

  //TODO !!!!!!!!!!!  
  while(currentTime != absTime)
  {
    //PTAD |= 0x04;
    currentTime = gPhyEventTriggerCmp;
    //PTAD &= ~0x04;
  }
  
  if(gPhyEventTriggerCtrl & gEventTimerCtrl_F_c)
  {
    gPhyEventTriggerCtrl |= gEventTimerCtrl_F_c;
    
  }
  gPhyEventTriggerCtrl |= (gEventTimerCtrl_IE_c);
  //PTAD |= (uint8_t )(0x02);
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/******************************************************************************
* Description : TPM2 Timer overflow flag 	
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/
void PhyTime_ISR(void)
{
  
  if((gPhyEventTriggerCtrl & gTPM_STATUS_CH0F_MASK_c) && (gPhyEventTriggerCtrl & gTPM_STATUS_CHIE_MASK_c))   //If channel Flag is set and if CHIE is enabled 
  {
//    gPhyEventTriggerCtrl |= gTPM_STATUS_CH0F_MASK_c;                          //@CMA OTA, Clear Flags in the next event PhyTimeEventTrigger_Handler(); //Clean flag for CH0
    PhyTimeEventTrigger_Handler();                                              //CH0 is the Event Trigger
  }
 
 else if ((gPhyEventTimeoutCtrl & gTPM_STATUS_CH1F_MASK_c) && (gPhyEventTimeoutCtrl & gTPM_STATUS_CHIE_MASK_c))         
  {
//    gPhyEventTimeoutCtrl |= gTPM_STATUS_CH1F_MASK_c;                          //@CMA OTA, Clear Flags in the next event PhyTimeEventTimeout_Handler();   //Clean flag for CH1. 
    PhyTimeEventTimeout_Handler();                                              //CH1 is the Event Timeout
  }
  else if (gPhyTimerCtrl & gTimerCtrl_TOF_c)
  {
    PhyTimeOverFlow_Handler();
  }
  

}




/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyTimeEventTrigger_Handler(void)
{
  
  if(gPhyEventTriggerCtrl & gEventTimerCtrl_F_c)                                //If Channel 0 Flag is set
  {
    gPhyEventTriggerCtrl |= gEventTimerCtrl_F_c;                                //Clean CH0F
    if(mPhyEventTriggerTimer.wordAccess[0] > 0)
    {
      mPhyEventTriggerTimer.wordAccess[0]--;
    }
    else
    {
      PhyTimeHandleEventTrigger();
      gPhyEventTriggerCtrl &= ~(gEventTimerCtrl_IE_c);                          //Disable CHIE
    }
  }
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/


void PhyTimeEventTimeout_Handler(void)
{

  if(gPhyEventTimeoutCtrl & gEventTimerCtrl_F_c)        
  {
    gPhyEventTimeoutCtrl |= gEventTimerCtrl_F_c;                                //Clean CH1F 
    if(mPhyEventTimeoutTimer.wordAccess[0] > 0)
    {
      mPhyEventTimeoutTimer.wordAccess[0]--;
    }
    else
    {
      gPhyEventTimeoutCtrl &= ~(gEventTimerCtrl_IE_c);                          //Disable CHIE
      PhyTimeHandleEventTimeout();                                             
    }
  }
  
}


/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyTimeOverFlow_Handler(void)
{
  
  if(gPhyTimerCtrl & gTimerCtrl_TOF_c)  //If TOF Flag is set
  {
    gPhyTimerCtrl |= gTimerCtrl_TOF_c;   
  }
  mPhyCurrentTime.wordAccess[0]++;

}
